/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpUtilities.c,v 1.13 2001/01/25 22:12:06 jeffc Exp $
____________________________________________________________________________*/
#include "pgpConfig.h"
#include "pgpErrors.h"
#include "pgpMem.h"
#include "pgpPFLPriv.h"

#include "pgpContext.h"
#include "pgpFileSpec.h"
#include "pgpUtilities.h"
#include "pgpUtilitiesPriv.h"
#include "pgpOptionList.h"
#include "pgpMacBinary.h"
#include "pgpMacFileMapping.h"
#include "pgpSDKPriv.h"
#include "pgpFIPSPriv.h"
#include "pgpKeyPriv.h"

#if PGP_MACINTOSH

#include "MacEnvirons.h"
#include "MacErrors.h"
#include "MacFiles.h"
#include "MacStrings.h"

#elif PGP_WIN32

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#endif



		


	PGPError
PGPCopyFileSpec(
	PGPFileSpecRef	fileRef,
	PGPFileSpecRef *outRef )
{
	PGPValidatePtr( outRef );
	*outRef	= NULL;
	PFLValidateFileSpec( (PFLConstFileSpecRef)fileRef );
	
	pgpEnterPGPErrorFunction();

	return( PFLCopyFileSpec( (PFLConstFileSpecRef)fileRef,
		(PFLFileSpecRef*)outRef ) );
}

	PGPError
PGPFreeFileSpec( PGPFileSpecRef		fileRef)
{
	PFLValidateFileSpec( (PFLFileSpecRef)fileRef );
	
	pgpEnterPGPErrorFunction();

	return( PFLFreeFileSpec( (PFLFileSpecRef)fileRef ) );
}

	PGPError
PGPRenameFile(
	PGPFileSpecRef 	fileRef,
	const char *	newName)
{
	PFLValidateFileSpec( (PFLFileSpecRef)fileRef );
	PGPValidatePtr( newName );
	
	pgpEnterPGPErrorFunction();

	return( PFLFileSpecRename( (PFLFileSpecRef)fileRef, newName ) );
}

	PGPError
PGPDeleteFile(PGPFileSpecRef fileRef)
{
	PFLValidateFileSpec( (PFLFileSpecRef)fileRef );
	
	pgpEnterPGPErrorFunction();

	return( PFLFileSpecDelete( (PFLFileSpecRef)fileRef ) );
}


#if PGP_MACINTOSH	/* [ */

	PGPError 
PGPNewFileSpecFromFSSpec(
	PGPContextRef		context,
	FSSpec const *		spec,
	PGPFileSpecRef *	outRef )
{
	return( PFLNewFileSpecFromFSSpec( PGPPeekContextMemoryMgr( context ),
		spec, (PFLFileSpecRef*)outRef ) );
}


	PGPError
PGPGetFSSpecFromFileSpec(
	PGPFileSpecRef	fileRef,
	FSSpec *		spec)
{
	pgpEnterPGPErrorFunction();

	return( PFLGetFSSpecFromFileSpec( (PFLFileSpecRef)fileRef, spec ) );
}

#else /* ] PGP_MACINTOSH [ */


	PGPError 
PGPNewFileSpecFromFullPath(
	PGPContextRef		context,
	char const *		path,
	PGPFileSpecRef *		outRef )
{
	PGPError	err	= kPGPError_NoErr;
	
	pgpEnterPGPErrorFunction();

	err	= PFLNewFileSpecFromFullPath( PGPPeekContextMemoryMgr( context ),
		path, (PFLFileSpecRef *)outRef );
	return err;
}



	PGPError 
PGPGetFullPathFromFileSpec(
	PGPFileSpecRef	fileRefIn,
	char **			fullPathPtr)
{
	PGPError			err;
	PGPMemoryMgrRef		memoryMgr	= NULL;
	char *				tempPath	= NULL;
	char *				fullPath	= NULL;
	PFLConstFileSpecRef	fileRef	= (PFLConstFileSpecRef)fileRefIn;
	
	PGPValidatePtr( fullPathPtr );
	*fullPathPtr	= NULL;
	PFLValidateFileSpec( fileRef );
	
	pgpEnterPGPErrorFunction();

	memoryMgr	= PFLGetFileSpecMemoryMgr( fileRef );
	
	err	= PFLGetFullPathFromFileSpec( fileRef, &tempPath );
	if ( IsntPGPError( err ) )
	{
		fullPath	= (char *)
			PGPNewData( memoryMgr, strlen( tempPath ) + 1, 0);
		if ( IsntNull( fullPath ) )
		{
			strcpy( fullPath, tempPath );
		}
		else
		{
			err	= kPGPError_OutOfMemory;
		}
		PGPFreeData( tempPath );
	}
	
	*fullPathPtr	= fullPath;
	return( err );
}



#endif	/* ] PGP_MACINTOSH */





/*____________________________________________________________________________
	Examine the input file to see if it's a MacBinary file.  If it is
	not a MacBinary file, then nothing is done.  Otherwise, it is
	converted, the original file is deleted and the resulting file is
	designated by 'outPGPSpec'.
	
	creator and type code pointers may be
	null but otherwise contain the mac creator and type.
	
	The output file may have a different name than the original because
	its Mac creator/type codes may be mapped into a file name extension.
	
	Example (assuming it's an MS-Word file):
		MyStuff.doc	=> MyStuff.doc
		MyStuff.bin => MyStuff.doc
		MyStuff		=> MyStuff.doc
____________________________________________________________________________*/
	PGPError
PGPMacBinaryToLocal(
	PGPFileSpecRef		inPGPSpec,
	PGPFileSpecRef *	outPGPSpec,
	PGPUInt32 *			macCreator,
	PGPUInt32 *			macType )
{
	PGPError			err;
	
	if ( IsntNull( macCreator ) )
		*macCreator	= 0;
	if ( IsntNull( macType ) )
		*macType	= 0;
	if ( IsntNull( outPGPSpec ) )
		*outPGPSpec	= NULL;
		
	PGPValidatePtr( outPGPSpec );
	PFLValidateFileSpec( (PFLFileSpecRef)inPGPSpec );
	
	pgpEnterPGPErrorFunction();

	err = pgpMacBinaryToLocal( (PFLFileSpecRef)inPGPSpec,
			(PFLFileSpecRef *)outPGPSpec, macCreator, macType );
	
	return( err );
}

	PGPError
pgpEnterFunction(
	const char 		*fileName,
	PGPUInt32 		lineNumber)
{
	PGPError	err = kPGPError_NoErr;
	
	(void) fileName;
	(void) lineNumber;

	pgpAssert( pgpsdkIsInited() );
	
	if( pgpFIPSModeEnabled() )
	{
		err = PGPGetSDKErrorState();
		if( IsPGPError( err ) )
		{
			pgpDebugMsg( "pgpEnterFunction: SDK is in error state" );
		}
		else if( pgpsdkIsInited() && ! pgpHaveRunAllSelfTests() )
		{
			pgpDebugMsg( "pgpEnterFunction: SDK self tests have not been executed" );
			err = kPGPError_SelfTestsNotExecuted;
		}
	}
	
	pgpKeyDBBackendUpdate();

	return( err );
}

#if PGP_MACINTOSH	/* [ */

static FSSpec		sLibraryFileSpec	= {0, };
static long			sLibraryFileID		= 0;

	OSStatus
SetLibraryFSSpec(const FSSpec *fileSpec)
{
	OSStatus	status;
	CInfoPBRec	cpb;
	
	// This routine should be called once ONLY from the library INIT routine.
	pgpAssert( sLibraryFileSpec.name[0] == 0 );
	pgpAssert( sLibraryFileID == 0 );
	
	status = FSpGetCatInfo( fileSpec, &cpb );
	if( IsntErr( status ) )
	{
		HParamBlockRec	pb;
		
		sLibraryFileSpec = *fileSpec;
		
		// Create a file ID so we can track this file later.
		
		pgpClearMemory( &pb, sizeof( pb ) );
		
		pb.fidParam.ioVRefNum	= sLibraryFileSpec.vRefNum;
		pb.fidParam.ioNamePtr	= sLibraryFileSpec.name;
		pb.fidParam.ioSrcDirID	= sLibraryFileSpec.parID;
		
		if( IsntErr( PBCreateFileIDRefSync( &pb ) ) )
		{
			sLibraryFileID = pb.fidParam.ioFileID;
		}
	}
	
	return( status );
}

	OSStatus
GetLibraryFSSpec(FSSpec *fileSpec)
{
	OSStatus	status;
	CInfoPBRec	cpb;
	
	pgpAssertAddrValid( fileSpec, FSSpec );
	AssertSpecIsValid( &sLibraryFileSpec, "PGPGetPGPLibraryFSSpec" );
	
	status = FSpGetCatInfo( &sLibraryFileSpec, &cpb );
	if( IsErr( status ) && sLibraryFileID != 0 )
	{
		// Find the library using the file ID.
		
		HParamBlockRec	pb;
		FSSpec			specCopy;
		
		pgpClearMemory( &pb, sizeof( pb ) );
		
		specCopy = sLibraryFileSpec;
		
		pb.fidParam.ioVRefNum	= specCopy.vRefNum;
		pb.fidParam.ioNamePtr	= specCopy.name;
		pb.fidParam.ioFileID	= sLibraryFileID;
		
		status = PBResolveFileIDRefSync( &pb );
		if( IsntErr( status ) )
		{
			specCopy.parID = pb.fidParam.ioSrcDirID;
			
			status = FSpGetCatInfo( &specCopy, &cpb );
			if( IsntErr( status ) )
			{
				sLibraryFileSpec = specCopy;
			}
		}
	}

	AssertNoErr( status, "GetPGPLibFileSpec" );
	
	*fileSpec = sLibraryFileSpec;
	
	return( status );
}


#endif	/* ] PGP_MACINTOSH */




/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
